<?php
/**
 * ChatReact Admin
 *
 * @package ChatReact
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * ChatReact Admin class
 */
class ChatReact_Admin {

    /**
     * Single instance of the class
     *
     * @var ChatReact_Admin|null
     */
    private static $instance = null;

    /**
     * Option name for widget assignments
     *
     * @var string
     */
    const OPTION_ASSIGNMENTS = 'chatreact_widget_assignments';

    /**
     * Option name for enabled post types in search
     *
     * @var string
     */
    const OPTION_ENABLED_POST_TYPES = 'chatreact_enabled_post_types';

    /**
     * Get single instance of the class
     *
     * @return ChatReact_Admin
     */
    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
        add_action( 'wp_ajax_chatreact_save_assignment', array( $this, 'ajax_save_assignment' ) );
        add_action( 'wp_ajax_chatreact_update_assignment', array( $this, 'ajax_update_assignment' ) );
        add_action( 'wp_ajax_chatreact_delete_assignment', array( $this, 'ajax_delete_assignment' ) );
        add_action( 'wp_ajax_chatreact_search_posts', array( $this, 'ajax_search_posts' ) );
        add_action( 'wp_ajax_chatreact_save_post_types', array( $this, 'ajax_save_post_types' ) );
        add_action( 'wp_ajax_chatreact_get_post_types', array( $this, 'ajax_get_post_types' ) );
        // Note: wp_footer hook for render_assigned_widgets is registered in chatreact.php
        // to ensure it works both in admin and frontend contexts
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        // ChatReact icon (40x40 PNG)
        $icon_url = CHATREACT_PLUGIN_URL . 'assets/images/icon-40.png';

        add_menu_page(
            __( 'ChatReact', 'chatreact' ),
            __( 'ChatReact', 'chatreact' ),
            'manage_options',
            'chatreact',
            array( $this, 'render_admin_page' ),
            $icon_url,
            30
        );
    }

    /**
     * Enqueue admin assets
     *
     * @param string $hook The current admin page
     */
    public function enqueue_admin_assets( $hook ) {
        // Menu icon styles - load on all admin pages
        wp_add_inline_style( 'wp-admin', '
            #adminmenu .toplevel_page_chatreact .wp-menu-image img {
                padding: 3px 0 0;
                opacity: 1;
                width: 28px;
            }
        ' );

        if ( 'toplevel_page_chatreact' !== $hook ) {
            return;
        }

        wp_enqueue_style(
            'chatreact-admin',
            CHATREACT_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            CHATREACT_VERSION
        );

        wp_enqueue_script(
            'chatreact-admin',
            CHATREACT_PLUGIN_URL . 'assets/js/admin.js',
            array( 'jquery' ),
            CHATREACT_VERSION,
            true
        );

        wp_localize_script( 'chatreact-admin', 'chatreactAdmin', array(
            'ajaxUrl' => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'chatreact_admin_nonce' ),
            'strings' => array(
                'confirmDelete' => __( 'Are you sure you want to delete this assignment?', 'chatreact' ),
                'saving' => __( 'Saving...', 'chatreact' ),
                'saved' => __( 'Saved successfully!', 'chatreact' ),
                'error' => __( 'An error occurred. Please try again.', 'chatreact' ),
                'selectPage' => __( 'Please select a page or post.', 'chatreact' ),
                'enterChatbotId' => __( 'Please enter a Chatbot ID.', 'chatreact' ),
                'selectPlaceholder' => __( 'Search pages and posts...', 'chatreact' ),
                'deleteBtn' => __( 'Delete', 'chatreact' ),
                'fillAllFields' => __( 'Please fill in all fields.', 'chatreact' ),
            ),
        ) );
    }

    /**
     * Get all widget assignments
     *
     * @return array
     */
    public function get_assignments() {
        return get_option( self::OPTION_ASSIGNMENTS, array() );
    }

    /**
     * Save widget assignments
     *
     * @param array $assignments The assignments to save
     * @return bool
     */
    public function save_assignments( $assignments ) {
        return update_option( self::OPTION_ASSIGNMENTS, $assignments );
    }

    /**
     * AJAX: Save a widget assignment
     */
    public function ajax_save_assignment() {
        check_ajax_referer( 'chatreact_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'chatreact' ) ) );
        }

        $post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : -1;
        $chatbot_id = isset( $_POST['chatbot_id'] ) ? sanitize_text_field( $_POST['chatbot_id'] ) : '';

        // post_id = 0 means "Everywhere", so we only reject negative values or when not provided
        if ( $post_id < 0 ) {
            wp_send_json_error( array( 'message' => __( 'Invalid page/post selected.', 'chatreact' ) ) );
        }

        if ( empty( $chatbot_id ) ) {
            wp_send_json_error( array( 'message' => __( 'Chatbot ID is required.', 'chatreact' ) ) );
        }

        $assignments = $this->get_assignments();
        
        // Generate unique ID for this assignment
        $assignment_id = 'assign_' . $post_id . '_' . time();
        
        // Check if an assignment for this post already exists
        foreach ( $assignments as $id => $assignment ) {
            if ( isset( $assignment['post_id'] ) && $assignment['post_id'] === $post_id ) {
                // Update existing assignment
                $assignment_id = $id;
                break;
            }
        }

        // Get exclude_urls (only relevant for "Everywhere" assignment)
        $exclude_urls = isset( $_POST['exclude_urls'] ) ? sanitize_textarea_field( $_POST['exclude_urls'] ) : '';

        $assignments[ $assignment_id ] = array(
            'post_id' => $post_id,
            'chatbot_id' => $chatbot_id,
            'exclude_urls' => $exclude_urls,
            'created_at' => current_time( 'mysql' ),
        );

        $this->save_assignments( $assignments );

        // Get post info for response
        if ( $post_id === 0 ) {
            // "Everywhere" assignment
            $post_title = __( 'Everywhere', 'chatreact' );
            $post_type_label = __( 'All pages', 'chatreact' );
        } else {
            $post = get_post( $post_id );
            $post_type_obj = get_post_type_object( $post->post_type );
            $post_title = $post->post_title;
            $post_type_label = $post_type_obj->labels->singular_name;
        }

        wp_send_json_success( array(
            'message' => __( 'Widget assignment saved successfully!', 'chatreact' ),
            'assignment' => array(
                'id' => $assignment_id,
                'post_id' => $post_id,
                'post_title' => $post_title,
                'post_type' => $post_type_label,
                'chatbot_id' => $chatbot_id,
                'exclude_urls' => $exclude_urls,
            ),
        ) );
    }

    /**
     * AJAX: Update a widget assignment
     */
    public function ajax_update_assignment() {
        check_ajax_referer( 'chatreact_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'chatreact' ) ) );
        }

        $assignment_id = isset( $_POST['assignment_id'] ) ? sanitize_text_field( $_POST['assignment_id'] ) : '';
        $chatbot_id = isset( $_POST['chatbot_id'] ) ? sanitize_text_field( $_POST['chatbot_id'] ) : '';

        if ( empty( $assignment_id ) ) {
            wp_send_json_error( array( 'message' => __( 'Invalid assignment.', 'chatreact' ) ) );
        }

        if ( empty( $chatbot_id ) ) {
            wp_send_json_error( array( 'message' => __( 'Chatbot ID is required.', 'chatreact' ) ) );
        }

        // Get exclude_urls if provided
        $exclude_urls = isset( $_POST['exclude_urls'] ) ? sanitize_textarea_field( $_POST['exclude_urls'] ) : null;

        $assignments = $this->get_assignments();
        
        if ( isset( $assignments[ $assignment_id ] ) ) {
            $assignments[ $assignment_id ]['chatbot_id'] = $chatbot_id;
            
            // Update exclude_urls if provided
            if ( $exclude_urls !== null ) {
                $assignments[ $assignment_id ]['exclude_urls'] = $exclude_urls;
            }
            
            $this->save_assignments( $assignments );
            wp_send_json_success( array( 'message' => __( 'Assignment updated successfully!', 'chatreact' ) ) );
        } else {
            wp_send_json_error( array( 'message' => __( 'Assignment not found.', 'chatreact' ) ) );
        }
    }

    /**
     * AJAX: Delete a widget assignment
     */
    public function ajax_delete_assignment() {
        check_ajax_referer( 'chatreact_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'chatreact' ) ) );
        }

        $assignment_id = isset( $_POST['assignment_id'] ) ? sanitize_text_field( $_POST['assignment_id'] ) : '';

        if ( empty( $assignment_id ) ) {
            wp_send_json_error( array( 'message' => __( 'Invalid assignment.', 'chatreact' ) ) );
        }

        $assignments = $this->get_assignments();
        
        if ( isset( $assignments[ $assignment_id ] ) ) {
            unset( $assignments[ $assignment_id ] );
            $this->save_assignments( $assignments );
            wp_send_json_success( array( 'message' => __( 'Assignment deleted successfully!', 'chatreact' ) ) );
        } else {
            wp_send_json_error( array( 'message' => __( 'Assignment not found.', 'chatreact' ) ) );
        }
    }

    /**
     * Render assigned widgets on frontend
     */
    public function render_assigned_widgets() {
        if ( is_admin() ) {
            return;
        }

        $current_post_id = get_queried_object_id();
        $current_url = $this->get_current_url_path();
        $assignments = $this->get_assignments();
        
        // First check for "Everywhere" assignment (post_id = 0)
        foreach ( $assignments as $assignment ) {
            if ( isset( $assignment['post_id'] ) && ( empty( $assignment['post_id'] ) || $assignment['post_id'] == 0 ) ) {
                // Check if current URL is in exclude list
                if ( $this->is_url_excluded( $current_url, $assignment ) ) {
                    return; // URL is excluded, don't render
                }
                
                // Found an "Everywhere" assignment - render the widget on all pages
                echo ChatReact::render_chat_widget(
                    $assignment['chatbot_id']
                );
                return; // "Everywhere" takes precedence
            }
        }
        
        // Then check for page-specific assignments
        if ( ! $current_post_id ) {
            return;
        }
        
        foreach ( $assignments as $assignment ) {
            if ( isset( $assignment['post_id'] ) && $assignment['post_id'] === $current_post_id ) {
                // Found a matching assignment - render the widget
                echo ChatReact::render_chat_widget(
                    $assignment['chatbot_id']
                );
                break; // Only one widget per page
            }
        }
    }

    /**
     * Get current URL path (without domain)
     *
     * @return string
     */
    private function get_current_url_path() {
        $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? $_SERVER['REQUEST_URI'] : '';
        // Remove query string
        $path = strtok( $request_uri, '?' );
        // Normalize: remove trailing slash (except for root)
        if ( $path !== '/' ) {
            $path = rtrim( $path, '/' );
        }
        return $path;
    }

    /**
     * Check if current URL should be excluded from widget display
     *
     * @param string $current_url The current URL path
     * @param array  $assignment  The assignment data
     * @return bool
     */
    private function is_url_excluded( $current_url, $assignment ) {
        if ( empty( $assignment['exclude_urls'] ) ) {
            return false;
        }

        // Split exclude_urls by newline and process each
        $exclude_patterns = array_filter( array_map( 'trim', explode( "\n", $assignment['exclude_urls'] ) ) );
        
        foreach ( $exclude_patterns as $pattern ) {
            // Skip empty lines
            if ( empty( $pattern ) ) {
                continue;
            }
            
            // Normalize pattern: remove domain if present, keep only path
            $pattern = $this->normalize_url_pattern( $pattern );
            
            // Check for wildcard matching
            if ( strpos( $pattern, '*' ) !== false ) {
                // Convert wildcard pattern to regex
                $regex_pattern = str_replace( 
                    array( '*', '/' ), 
                    array( '.*', '\/' ), 
                    $pattern 
                );
                if ( preg_match( '/^' . $regex_pattern . '$/i', $current_url ) ) {
                    return true;
                }
            } else {
                // Exact match (case-insensitive)
                $pattern_normalized = ( $pattern !== '/' ) ? rtrim( $pattern, '/' ) : $pattern;
                if ( strcasecmp( $current_url, $pattern_normalized ) === 0 ) {
                    return true;
                }
            }
        }
        
        return false;
    }

    /**
     * Normalize URL pattern to path only
     *
     * @param string $pattern The URL pattern
     * @return string
     */
    private function normalize_url_pattern( $pattern ) {
        // If it's a full URL, extract just the path
        if ( preg_match( '/^https?:\/\/[^\/]+(.*)$/i', $pattern, $matches ) ) {
            $pattern = $matches[1];
        }
        
        // Ensure it starts with /
        if ( strpos( $pattern, '/' ) !== 0 && strpos( $pattern, '*' ) !== 0 ) {
            $pattern = '/' . $pattern;
        }
        
        return $pattern;
    }

    /**
     * Get enabled post types for search
     * Defaults to only 'page' if not set
     *
     * @return array
     */
    public function get_enabled_post_types() {
        $enabled = get_option( self::OPTION_ENABLED_POST_TYPES, array( 'page' ) );
        if ( empty( $enabled ) || ! is_array( $enabled ) ) {
            return array( 'page' );
        }
        return $enabled;
    }

    /**
     * Save enabled post types
     *
     * @param array $post_types The post types to enable
     * @return bool
     */
    public function save_enabled_post_types( $post_types ) {
        return update_option( self::OPTION_ENABLED_POST_TYPES, $post_types );
    }

    /**
     * AJAX: Get available post types and their enabled state
     */
    public function ajax_get_post_types() {
        check_ajax_referer( 'chatreact_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'chatreact' ) ) );
        }

        $enabled = $this->get_enabled_post_types();
        $all_post_types = get_post_types( array( 'public' => true ), 'objects' );
        $result = array();

        foreach ( $all_post_types as $post_type ) {
            // Skip 'attachment' post type
            if ( $post_type->name === 'attachment' ) {
                continue;
            }
            $result[] = array(
                'name' => $post_type->name,
                'label' => $post_type->labels->name,
                'enabled' => in_array( $post_type->name, $enabled, true ),
            );
        }

        wp_send_json_success( array( 'post_types' => $result ) );
    }

    /**
     * AJAX: Save enabled post types
     */
    public function ajax_save_post_types() {
        check_ajax_referer( 'chatreact_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'chatreact' ) ) );
        }

        $post_types = isset( $_POST['post_types'] ) ? $_POST['post_types'] : array();
        
        // Sanitize
        if ( is_string( $post_types ) ) {
            $post_types = json_decode( stripslashes( $post_types ), true );
        }
        
        if ( ! is_array( $post_types ) ) {
            $post_types = array();
        }

        $post_types = array_map( 'sanitize_text_field', $post_types );

        // Always ensure at least 'page' is enabled
        if ( empty( $post_types ) ) {
            $post_types = array( 'page' );
        }

        $this->save_enabled_post_types( $post_types );

        wp_send_json_success( array( 'message' => __( 'Post type settings saved!', 'chatreact' ) ) );
    }

    /**
     * AJAX: Search posts for dropdown
     * Uses efficient database query with LIMIT for large sites
     * Only searches enabled post types
     */
    public function ajax_search_posts() {
        check_ajax_referer( 'chatreact_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied.', 'chatreact' ) ) );
        }

        $search = isset( $_POST['search'] ) ? sanitize_text_field( $_POST['search'] ) : '';
        $results = array();

        // Minimum 2 characters for search, or empty to show recent posts
        if ( strlen( $search ) < 2 && ! empty( $search ) ) {
            wp_send_json_success( array( 'posts' => array() ) );
        }

        // Get only enabled post types
        $enabled_post_types = $this->get_enabled_post_types();

        // Build query args
        $args = array(
            'post_type' => $enabled_post_types,
            'posts_per_page' => 50, // Limit to 50 results for performance
            'post_status' => 'publish',
            'orderby' => 'relevance',
            'order' => 'DESC',
        );

        // Add search term if provided
        if ( ! empty( $search ) ) {
            $args['s'] = $search;
        } else {
            // If no search, show recent posts
            $args['orderby'] = 'date';
            $args['posts_per_page'] = 20;
        }

        $query = new WP_Query( $args );

        if ( $query->have_posts() ) {
            while ( $query->have_posts() ) {
                $query->the_post();
                $post_type_obj = get_post_type_object( get_post_type() );
                $results[] = array(
                    'id' => get_the_ID(),
                    'title' => get_the_title(),
                    'type' => $post_type_obj->labels->singular_name,
                    'type_slug' => get_post_type(),
                );
            }
            wp_reset_postdata();
        }

        wp_send_json_success( array( 'posts' => $results ) );
    }

    /**
     * Get all pages and posts for dropdown
     * DEPRECATED: This method loads all posts and causes memory issues on large sites.
     * Use ajax_search_posts() instead for AJAX-based search.
     *
     * @return array
     */
    public function get_all_posts_for_dropdown() {
        // Return empty array - use AJAX search instead
        // This prevents memory exhaustion on sites with many posts
        return array();
    }

    /**
     * Render the admin page
     */
    public function render_admin_page() {
        $assignments = $this->get_assignments();
        // Note: Posts are now loaded via AJAX search to prevent memory issues on large sites
        include CHATREACT_PLUGIN_DIR . 'admin/views/admin-page.php';
    }
}
