/**
 * ChatReact Admin JavaScript
 */
(function($) {
    'use strict';

    // Tab switching
    function initTabs() {
        $('.chatreact-tab').on('click', function(e) {
            e.preventDefault();
            
            var tabId = $(this).data('tab');
            
            // Update tab buttons
            $('.chatreact-tab').removeClass('active');
            $(this).addClass('active');
            
            // Update tab content
            $('.chatreact-tab-content').removeClass('active');
            $('#tab-' + tabId).addClass('active');
            
            // Save to localStorage
            localStorage.setItem('chatreact_active_tab', tabId);
        });
        
        // Restore last active tab
        var lastTab = localStorage.getItem('chatreact_active_tab');
        if (lastTab) {
            $('.chatreact-tab[data-tab="' + lastTab + '"]').click();
        }
    }

    // Show notice
    function showNotice(type, message) {
        var $notice = $('#chatreact-notice');
        $notice
            .removeClass('chatreact-notice-success chatreact-notice-error')
            .addClass('chatreact-notice-' + type)
            .addClass('show')
            .find('.chatreact-notice-message').text(message);
        
        setTimeout(function() {
            $notice.removeClass('show');
        }, 5000);
    }

    // Save assignment
    function initSaveAssignment() {
        $('#chatreact-add-assignment-form').on('submit', function(e) {
            e.preventDefault();
            
            var $form = $(this);
            var $btn = $form.find('.chatreact-btn-primary');
            var originalText = $btn.html();
            
            var postId = $form.find('[name="post_id"]').val();
            var chatbotId = $form.find('[name="chatbot_id"]').val();
            
            if (!postId) {
                showNotice('error', chatreactAdmin.strings.selectPage);
                return;
            }
            
            if (!chatbotId) {
                showNotice('error', chatreactAdmin.strings.enterChatbotId);
                return;
            }
            
            $btn.prop('disabled', true).html('<span class="chatreact-spinner"></span> ' + chatreactAdmin.strings.saving);
            
            $.ajax({
                url: chatreactAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'chatreact_save_assignment',
                    nonce: chatreactAdmin.nonce,
                    post_id: postId,
                    chatbot_id: chatbotId
                },
                success: function(response) {
                    if (response.success) {
                        showNotice('success', response.data.message);
                        
                        // Update the table
                        updateAssignmentsTable(response.data.assignment);
                        
                        // Reset form
                        $form.find('[name="chatbot_id"]').val('');
                        if (typeof window.resetPostSelect === 'function') {
                            window.resetPostSelect();
                        }
                    } else {
                        showNotice('error', response.data.message);
                    }
                },
                error: function() {
                    showNotice('error', chatreactAdmin.strings.error);
                },
                complete: function() {
                    $btn.prop('disabled', false).html(originalText);
                }
            });
        });
    }

    // Update assignments table after save
    function updateAssignmentsTable(assignment) {
        var $table = $('#chatreact-assignments-table');
        var $tbody = $table.find('tbody');
        var $emptyState = $('#chatreact-empty-state');
        
        // Hide empty state if visible
        $emptyState.hide();
        $table.show();
        
        // Check if row already exists
        var $existingRow = $tbody.find('tr[data-id="' + assignment.id + '"]');
        
        var rowHtml = buildAssignmentRow(assignment);
        
        if ($existingRow.length) {
            $existingRow.replaceWith(rowHtml);
        } else {
            $tbody.prepend(rowHtml);
        }
        
        // Update badge count
        var count = $tbody.find('tr').length;
        $('.chatreact-tab[data-tab="chat"] .chatreact-tab-badge').text(count);
    }

    // Build assignment row HTML
    function buildAssignmentRow(assignment) {
        var isEverywhere = assignment.post_id === 0 || assignment.post_id === '0';
        var iconClass = isEverywhere ? 'chatreact-page-icon chatreact-page-icon-everywhere' : 'chatreact-page-icon';
        var icon = isEverywhere 
            ? '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"></circle><line x1="2" y1="12" x2="22" y2="12"></line><path d="M12 2a15.3 15.3 0 0 1 4 10 15.3 15.3 0 0 1-4 10 15.3 15.3 0 0 1-4-10 15.3 15.3 0 0 1 4-10z"></path></svg>'
            : '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path><polyline points="14 2 14 8 20 8"></polyline></svg>';
        
        // Build exclude info for "Everywhere" assignments
        var excludeInfo = '';
        if (isEverywhere && assignment.exclude_urls) {
            var excludeCount = assignment.exclude_urls.split('\n').filter(function(url) { return url.trim() !== ''; }).length;
            if (excludeCount > 0) {
                excludeInfo = '<div class="chatreact-exclude-info">' +
                    '<svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"></circle><line x1="4.93" y1="4.93" x2="19.07" y2="19.07"></line></svg>' +
                    excludeCount + (excludeCount === 1 ? ' URL excluded' : ' URLs excluded') +
                '</div>';
            }
        }
        
        var excludeUrls = assignment.exclude_urls || '';
        
        return '<tr data-id="' + assignment.id + '" data-is-everywhere="' + (isEverywhere ? '1' : '0') + '">' +
            '<td>' +
                '<div class="chatreact-assignment-page">' +
                    '<div class="' + iconClass + '">' +
                        icon +
                    '</div>' +
                    '<div>' +
                        '<div class="chatreact-page-title">' + escapeHtml(assignment.post_title) + '</div>' +
                        '<div class="chatreact-page-type">' + escapeHtml(assignment.post_type) + '</div>' +
                        excludeInfo +
                    '</div>' +
                '</div>' +
            '</td>' +
            '<td><span class="chatreact-badge chatreact-badge-purple chatreact-chatbot-id">' + escapeHtml(assignment.chatbot_id) + '</span></td>' +
            '<td>' +
                '<div class="chatreact-actions">' +
                    '<button type="button" class="chatreact-btn chatreact-btn-ghost chatreact-btn-sm chatreact-edit-assignment" data-id="' + assignment.id + '" data-chatbot-id="' + escapeHtml(assignment.chatbot_id) + '" data-exclude-urls="' + escapeHtml(excludeUrls) + '" data-is-everywhere="' + (isEverywhere ? '1' : '0') + '" title="Edit">' +
                        '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"></path><path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"></path></svg>' +
                    '</button>' +
                    '<button type="button" class="chatreact-btn chatreact-btn-ghost chatreact-btn-sm chatreact-delete-assignment" data-id="' + assignment.id + '" title="Delete">' +
                        '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="3 6 5 6 21 6"></polyline><path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path></svg>' +
                    '</button>' +
                '</div>' +
            '</td>' +
        '</tr>';
    }

    // Escape HTML
    function escapeHtml(text) {
        var div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    // Edit assignment
    function initEditAssignment() {
        $(document).on('click', '.chatreact-edit-assignment', function(e) {
            e.preventDefault();
            
            var $btn = $(this);
            var $row = $btn.closest('tr');
            var assignmentId = $btn.data('id');
            var currentChatbotId = $btn.data('chatbot-id');
            var currentExcludeUrls = $btn.data('exclude-urls') || '';
            var isEverywhere = $btn.data('is-everywhere') === 1 || $btn.data('is-everywhere') === '1';
            
            // Check if already editing
            if ($row.hasClass('chatreact-edit-row')) {
                return;
            }
            
            // Get the chatbot ID cell
            var $chatbotCell = $row.find('.chatreact-chatbot-id').parent();
            var originalContent = $chatbotCell.html();
            
            // Build edit form HTML
            var editFormHtml = '<div class="chatreact-inline-edit">' +
                    '<input type="text" class="chatreact-input chatreact-edit-input" value="' + escapeHtml(currentChatbotId) + '" />' +
                    '<button type="button" class="chatreact-btn chatreact-btn-primary chatreact-btn-sm chatreact-save-edit" data-id="' + assignmentId + '" data-is-everywhere="' + (isEverywhere ? '1' : '0') + '">' +
                        '<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="20 6 9 17 4 12"></polyline></svg>' +
                    '</button>' +
                    '<button type="button" class="chatreact-btn chatreact-btn-ghost chatreact-btn-sm chatreact-cancel-edit">' +
                        '<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="18" y1="6" x2="6" y2="18"></line><line x1="6" y1="6" x2="18" y2="18"></line></svg>' +
                    '</button>' +
                '</div>';
            
            // Add exclude URLs field for "Everywhere" assignments
            if (isEverywhere) {
                editFormHtml += '<div class="chatreact-edit-exclude-urls">' +
                    '<label class="chatreact-exclude-urls-label">' +
                        '<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"></circle><line x1="4.93" y1="4.93" x2="19.07" y2="19.07"></line></svg>' +
                        ' Exclude URLs' +
                    '</label>' +
                    '<textarea class="chatreact-input chatreact-edit-exclude-input" rows="3" placeholder="/contact&#10;/checkout&#10;/admin/*">' + escapeHtml(currentExcludeUrls) + '</textarea>' +
                    '<p class="chatreact-exclude-urls-hint">One URL per line. Use * for wildcards.</p>' +
                '</div>';
            }
            
            // Replace with inline edit form
            $row.addClass('chatreact-edit-row');
            $chatbotCell.html(editFormHtml);
            
            // Store original content
            $row.data('original-content', originalContent);
            
            // Focus input
            $chatbotCell.find('.chatreact-edit-input').focus().select();
        });
        
        // Cancel edit
        $(document).on('click', '.chatreact-cancel-edit', function(e) {
            e.preventDefault();
            
            var $row = $(this).closest('tr');
            var $chatbotCell = $row.find('.chatreact-inline-edit').parent();
            var originalContent = $row.data('original-content');
            
            $row.removeClass('chatreact-edit-row');
            $chatbotCell.html(originalContent);
        });
        
        // Save edit
        $(document).on('click', '.chatreact-save-edit', function(e) {
            e.preventDefault();
            
            var $btn = $(this);
            var $row = $btn.closest('tr');
            var assignmentId = $btn.data('id');
            var isEverywhere = $btn.data('is-everywhere') === 1 || $btn.data('is-everywhere') === '1';
            var newChatbotId = $row.find('.chatreact-edit-input').val().trim();
            var newExcludeUrls = isEverywhere ? ($row.find('.chatreact-edit-exclude-input').val() || '') : '';
            
            if (!newChatbotId) {
                showNotice('error', chatreactAdmin.strings.enterChatbotId);
                return;
            }
            
            $btn.prop('disabled', true).html('<span class="chatreact-spinner"></span>');
            
            var requestData = {
                action: 'chatreact_update_assignment',
                nonce: chatreactAdmin.nonce,
                assignment_id: assignmentId,
                chatbot_id: newChatbotId
            };
            
            // Include exclude_urls for "Everywhere" assignments
            if (isEverywhere) {
                requestData.exclude_urls = newExcludeUrls;
            }
            
            $.ajax({
                url: chatreactAdmin.ajaxUrl,
                type: 'POST',
                data: requestData,
                success: function(response) {
                    if (response.success) {
                        showNotice('success', response.data.message);
                        
                        // Update the row
                        var $chatbotCell = $row.find('.chatreact-inline-edit').parent();
                        $row.removeClass('chatreact-edit-row');
                        $chatbotCell.html('<span class="chatreact-badge chatreact-badge-purple chatreact-chatbot-id">' + escapeHtml(newChatbotId) + '</span>');
                        
                        // Update the edit button data
                        var $editBtn = $row.find('.chatreact-edit-assignment');
                        $editBtn.data('chatbot-id', newChatbotId);
                        
                        // Update exclude URLs data and info display for "Everywhere" assignments
                        if (isEverywhere) {
                            $editBtn.data('exclude-urls', newExcludeUrls);
                            
                            // Update the exclude info display
                            var $excludeInfo = $row.find('.chatreact-exclude-info');
                            var excludeCount = newExcludeUrls.split('\n').filter(function(url) { return url.trim() !== ''; }).length;
                            
                            if (excludeCount > 0) {
                                var infoHtml = '<svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"></circle><line x1="4.93" y1="4.93" x2="19.07" y2="19.07"></line></svg>' +
                                    excludeCount + (excludeCount === 1 ? ' URL excluded' : ' URLs excluded');
                                
                                if ($excludeInfo.length) {
                                    $excludeInfo.html(infoHtml);
                                } else {
                                    $row.find('.chatreact-page-type').after('<div class="chatreact-exclude-info">' + infoHtml + '</div>');
                                }
                            } else {
                                $excludeInfo.remove();
                            }
                        }
                    } else {
                        showNotice('error', response.data.message);
                    }
                },
                error: function() {
                    showNotice('error', chatreactAdmin.strings.error);
                },
                complete: function() {
                    $btn.prop('disabled', false);
                }
            });
        });
        
        // Handle Enter key in edit input
        $(document).on('keydown', '.chatreact-edit-input', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                $(this).closest('.chatreact-inline-edit').find('.chatreact-save-edit').click();
            } else if (e.key === 'Escape') {
                e.preventDefault();
                $(this).closest('.chatreact-inline-edit').find('.chatreact-cancel-edit').click();
            }
        });
    }

    // Delete Modal
    var deleteModal = {
        $modal: null,
        $btn: null,
        $row: null,
        assignmentId: null,
        
        init: function() {
            var self = this;
            this.$modal = $('#chatreact-delete-modal');
            
            // Cancel button
            this.$modal.on('click', '#chatreact-modal-cancel', function() {
                self.close();
            });
            
            // Backdrop click
            this.$modal.on('click', '.chatreact-modal-backdrop', function() {
                self.close();
            });
            
            // Confirm button
            this.$modal.on('click', '#chatreact-modal-confirm', function() {
                self.confirmDelete();
            });
            
            // ESC key
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape' && self.$modal.hasClass('open')) {
                    self.close();
                }
            });
        },
        
        open: function($btn, assignmentId, $row) {
            this.$btn = $btn;
            this.assignmentId = assignmentId;
            this.$row = $row;
            this.$modal.addClass('open');
            $('body').css('overflow', 'hidden');
        },
        
        close: function() {
            this.$modal.removeClass('open');
            $('body').css('overflow', '');
            this.$btn = null;
            this.$row = null;
            this.assignmentId = null;
        },
        
        confirmDelete: function() {
            var self = this;
            var $confirmBtn = this.$modal.find('#chatreact-modal-confirm');
            
            $confirmBtn.prop('disabled', true).html(
                '<span class="chatreact-spinner"></span>'
            );
            
            $.ajax({
                url: chatreactAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'chatreact_delete_assignment',
                    nonce: chatreactAdmin.nonce,
                    assignment_id: self.assignmentId
                },
                success: function(response) {
                    if (response.success) {
                        self.$row.fadeOut(300, function() {
                            self.$row.remove();
                            
                            // Check if table is now empty
                            var $tbody = $('#chatreact-assignments-table tbody');
                            if ($tbody.find('tr').length === 0) {
                                $('#chatreact-assignments-table').hide();
                                $('#chatreact-empty-state').show();
                            }
                            
                            // Update badge count
                            var count = $tbody.find('tr').length;
                            $('.chatreact-tab[data-tab="chat"] .chatreact-tab-badge').text(count);
                        });
                        
                        self.close();
                        showNotice('success', response.data.message);
                    } else {
                        showNotice('error', response.data.message);
                    }
                },
                error: function() {
                    showNotice('error', chatreactAdmin.strings.error);
                },
                complete: function() {
                    $confirmBtn.prop('disabled', false).html(
                        '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="3 6 5 6 21 6"></polyline><path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path></svg>' +
                        chatreactAdmin.strings.deleteBtn
                    );
                }
            });
        }
    };

    // Delete assignment
    function initDeleteAssignment() {
        deleteModal.init();
        
        $(document).on('click', '.chatreact-delete-assignment', function(e) {
            e.preventDefault();
            
            var $btn = $(this);
            var assignmentId = $btn.data('id');
            var $row = $btn.closest('tr');
            
            deleteModal.open($btn, assignmentId, $row);
        });
    }

    // Custom searchable select
    function initSearchableSelect() {
        var $wrapper = $('#chatreact-post-select');
        var $trigger = $wrapper.find('.chatreact-select-trigger');
        var $dropdown = $wrapper.find('.chatreact-select-dropdown');
        var $searchInput = $dropdown.find('.chatreact-select-search input');
        var $options = $dropdown.find('.chatreact-select-option');
        var $groups = $dropdown.find('.chatreact-select-group');
        var $noResults = $dropdown.find('.chatreact-select-no-results');
        var $hiddenInput = $wrapper.find('input[type="hidden"]');
        
        // Toggle dropdown
        $trigger.on('click', function(e) {
            e.stopPropagation();
            var isOpen = $dropdown.hasClass('open');
            
            // Close all other dropdowns
            $('.chatreact-select-dropdown').removeClass('open');
            $('.chatreact-select-trigger').removeClass('open');
            
            if (!isOpen) {
                $dropdown.addClass('open');
                $trigger.addClass('open');
                $searchInput.val('').focus();
                filterOptions('');
            }
        });
        
        // Keyboard navigation on trigger
        $trigger.on('keydown', function(e) {
            if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                $trigger.click();
            }
        });
        
        // Close on outside click
        $(document).on('click', function(e) {
            if (!$(e.target).closest('.chatreact-select-wrapper').length) {
                $dropdown.removeClass('open');
                $trigger.removeClass('open');
            }
        });
        
        // Search functionality
        $searchInput.on('input', function() {
            var query = $(this).val().toLowerCase();
            filterOptions(query);
        });
        
        // Prevent dropdown close when clicking inside
        $dropdown.on('click', function(e) {
            e.stopPropagation();
        });
        
        // Filter options
        function filterOptions(query) {
            var hasResults = false;
            
            $groups.each(function() {
                var $group = $(this);
                var $groupOptions = $group.find('.chatreact-select-option');
                var groupHasVisible = false;
                
                $groupOptions.each(function() {
                    var $option = $(this);
                    var title = $option.data('title').toLowerCase();
                    var matches = title.indexOf(query) !== -1;
                    
                    if (matches) {
                        $option.show();
                        groupHasVisible = true;
                        hasResults = true;
                    } else {
                        $option.hide();
                    }
                });
                
                if (groupHasVisible) {
                    $group.show();
                } else {
                    $group.hide();
                }
            });
            
            if (hasResults) {
                $noResults.hide();
            } else {
                $noResults.show();
            }
        }
        
        // Select option
        $options.on('click', function() {
            var $option = $(this);
            var value = $option.data('value');
            var title = $option.data('title');
            var type = $option.data('type');
            
            // Update hidden input
            $hiddenInput.val(value);
            
            // Update trigger display
            $trigger.html(
                '<span class="selected-value">' +
                    '<span>' + escapeHtml(title) + '</span>' +
                    '<span class="selected-type">' + escapeHtml(type) + '</span>' +
                '</span>' +
                '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="6 9 12 15 18 9"></polyline></svg>'
            );
            
            // Mark as selected
            $options.removeClass('selected');
            $option.addClass('selected');
            
            // Close dropdown
            $dropdown.removeClass('open');
            $trigger.removeClass('open');
        });
        
        // Reset select
        window.resetPostSelect = function() {
            $hiddenInput.val('');
            $trigger.html(
                '<span class="placeholder">' + chatreactAdmin.strings.selectPlaceholder + '</span>' +
                '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="6 9 12 15 18 9"></polyline></svg>'
            );
            $options.removeClass('selected');
        };
    }

    // Add new row functionality
    function initAddRow() {
        var $addBtn = $('#chatreact-add-row-btn');
        var $saveBtn = $('#chatreact-save-all-btn');
        var $template = $('#chatreact-new-row-template');
        var $tbody = $('#chatreact-assignments-table tbody');
        var $table = $('#chatreact-assignments-table');
        var $emptyState = $('#chatreact-empty-state');
        
        // Add row button click
        $addBtn.on('click', function() {
            // Show table if hidden
            $table.show();
            $emptyState.hide();
            
            // Clone template and add to table
            var templateHtml = $template.html();
            var $newRow = $(templateHtml);
            $tbody.append($newRow);
            
            // Initialize inline select for new row
            initInlineSelectForRow($newRow);
            
            // Show save button
            updateSaveButtonVisibility();
            
            // Focus on the first input
            $newRow.find('.chatreact-inline-select-trigger').first().focus();
        });
        
        // Remove new row
        $(document).on('click', '.chatreact-remove-new-row', function() {
            var $row = $(this).closest('tr');
            $row.fadeOut(200, function() {
                $row.remove();
                updateSaveButtonVisibility();
                
                // Check if table is empty
                if ($tbody.find('tr').length === 0) {
                    $table.hide();
                    $emptyState.show();
                }
            });
        });
        
        // Update save button visibility
        function updateSaveButtonVisibility() {
            var hasNewRows = $tbody.find('.chatreact-new-row').length > 0;
            if (hasNewRows) {
                $saveBtn.show();
            } else {
                $saveBtn.hide();
            }
        }
    }
    
    // Initialize inline select for a specific row with AJAX search
    function initInlineSelectForRow($row) {
        var $wrapper = $row.find('.chatreact-inline-select-wrapper');
        var $trigger = $wrapper.find('.chatreact-inline-select-trigger');
        var $dropdown = $wrapper.find('.chatreact-inline-select-dropdown');
        var $searchInput = $dropdown.find('.chatreact-inline-select-search input');
        var $hiddenInput = $wrapper.find('.chatreact-row-post-id');
        var $ajaxResults = $wrapper.find('.chatreact-inline-select-ajax-results');
        var $loading = $wrapper.find('.chatreact-inline-select-loading');
        var $noResults = $wrapper.find('.chatreact-inline-select-no-results');
        var $hint = $wrapper.find('.chatreact-inline-select-hint');
        var searchTimeout = null;
        var currentRequest = null;
        
        // Toggle dropdown
        $trigger.on('click', function(e) {
            e.stopPropagation();
            var isOpen = $dropdown.hasClass('open');
            
            // Close all other dropdowns
            $('.chatreact-inline-select-dropdown').removeClass('open');
            $('.chatreact-inline-select-trigger').removeClass('open');
            
            if (!isOpen) {
                $dropdown.addClass('open');
                $trigger.addClass('open');
                $searchInput.val('').focus();
                $hint.show();
                $noResults.hide();
                $loading.hide();
                $ajaxResults.html('');
                
                // Show "Everywhere" option
                $wrapper.find('.chatreact-inline-select-option-everywhere').show();
            }
        });
        
        // AJAX Search with debounce
        $searchInput.on('input', function() {
            var query = $(this).val().trim();
            
            // Clear previous timeout
            if (searchTimeout) {
                clearTimeout(searchTimeout);
            }
            
            // Cancel previous request
            if (currentRequest) {
                currentRequest.abort();
            }
            
            // Filter "Everywhere" option locally
            var $everywhere = $wrapper.find('.chatreact-inline-select-option-everywhere');
            var everywhereTitle = ($everywhere.data('title') || '').toLowerCase();
            $everywhere.toggle(query === '' || everywhereTitle.indexOf(query.toLowerCase()) !== -1);
            
            // Show hint if less than 2 characters
            if (query.length < 2) {
                $hint.show();
                $noResults.hide();
                $loading.hide();
                $ajaxResults.html('');
                return;
            }
            
            $hint.hide();
            
            // Debounce the search
            searchTimeout = setTimeout(function() {
                $loading.show();
                $noResults.hide();
                $ajaxResults.html('');
                
                currentRequest = $.ajax({
                    url: chatreactAdmin.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'chatreact_search_posts',
                        nonce: chatreactAdmin.nonce,
                        search: query
                    },
                    success: function(response) {
                        $loading.hide();
                        
                        if (response.success && response.data.posts.length > 0) {
                            var html = '';
                            var groupedPosts = {};
                            
                            // Group posts by type
                            response.data.posts.forEach(function(post) {
                                if (!groupedPosts[post.type_slug]) {
                                    groupedPosts[post.type_slug] = {
                                        label: post.type,
                                        items: []
                                    };
                                }
                                groupedPosts[post.type_slug].items.push(post);
                            });
                            
                            // Build HTML
                            for (var typeSlug in groupedPosts) {
                                var group = groupedPosts[typeSlug];
                                html += '<div class="chatreact-inline-select-group">';
                                html += '<div class="chatreact-inline-select-group-label">' + escapeHtml(group.label) + '</div>';
                                
                                group.items.forEach(function(item) {
                                    html += '<div class="chatreact-inline-select-option chatreact-ajax-option" data-value="' + item.id + '" data-title="' + escapeHtml(item.title) + '" data-type="' + escapeHtml(group.label) + '">';
                                    html += '<div class="option-icon">';
                                    html += '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path><polyline points="14 2 14 8 20 8"></polyline></svg>';
                                    html += '</div>';
                                    html += '<div class="option-content">';
                                    html += '<div class="option-title">' + escapeHtml(item.title) + '</div>';
                                    html += '<div class="option-type">' + escapeHtml(group.label) + '</div>';
                                    html += '</div>';
                                    html += '</div>';
                                });
                                
                                html += '</div>';
                            }
                            
                            $ajaxResults.html(html);
                            $noResults.hide();
                        } else {
                            $ajaxResults.html('');
                            $noResults.show();
                        }
                    },
                    error: function(xhr, status) {
                        if (status !== 'abort') {
                            $loading.hide();
                            $noResults.show();
                        }
                    }
                });
            }, 300); // 300ms debounce
        });
        
        // Select option (delegated for AJAX-loaded options)
        $dropdown.on('click', '.chatreact-inline-select-option', function() {
            var $option = $(this);
            var value = $option.data('value');
            var title = $option.data('title');
            var isEverywhere = (value === 0 || value === '0');
            
            $hiddenInput.val(value);
            $trigger.html(
                '<span>' + escapeHtml(title) + '</span>' +
                '<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="6 9 12 15 18 9"></polyline></svg>'
            );
            
            $dropdown.find('.chatreact-inline-select-option').removeClass('selected');
            $option.addClass('selected');
            
            $dropdown.removeClass('open');
            $trigger.removeClass('open');
            
            // Show/hide exclude URLs field based on selection
            var $excludeWrapper = $row.find('.chatreact-exclude-urls-wrapper');
            if (isEverywhere) {
                $excludeWrapper.slideDown(200);
            } else {
                $excludeWrapper.slideUp(200);
                $excludeWrapper.find('.chatreact-row-exclude-urls').val('');
            }
        });
        
        // Close on outside click
        $(document).on('click', function(e) {
            if (!$(e.target).closest('.chatreact-inline-select-wrapper').length) {
                $dropdown.removeClass('open');
                $trigger.removeClass('open');
            }
        });
    }
    
    // Save all new assignments
    function initSaveAll() {
        var $saveBtn = $('#chatreact-save-all-btn');
        var $tbody = $('#chatreact-assignments-table tbody');
        
        $saveBtn.on('click', function() {
            var $newRows = $tbody.find('.chatreact-new-row');
            var assignments = [];
            var hasErrors = false;
            
            // Collect all new assignments
            $newRows.each(function() {
                var $row = $(this);
                var postId = $row.find('.chatreact-row-post-id').val();
                var chatbotId = $row.find('.chatreact-row-chatbot-id').val();
                var excludeUrls = $row.find('.chatreact-row-exclude-urls').val() || '';
                
                if (!postId) {
                    $row.find('.chatreact-inline-select-trigger').css('border-color', '#ef4444');
                    hasErrors = true;
                } else {
                    $row.find('.chatreact-inline-select-trigger').css('border-color', '');
                }
                
                if (!chatbotId) {
                    $row.find('.chatreact-row-chatbot-id').css('border-color', '#ef4444');
                    hasErrors = true;
                } else {
                    $row.find('.chatreact-row-chatbot-id').css('border-color', '');
                }
                
                if (postId && chatbotId) {
                    assignments.push({
                        post_id: postId,
                        chatbot_id: chatbotId,
                        exclude_urls: excludeUrls,
                        $row: $row
                    });
                }
            });
            
            if (hasErrors) {
                showNotice('error', chatreactAdmin.strings.fillAllFields || 'Please fill in all fields.');
                return;
            }
            
            if (assignments.length === 0) {
                return;
            }
            
            // Disable button
            var originalHtml = $saveBtn.html();
            $saveBtn.prop('disabled', true).html('<span class="chatreact-spinner"></span> ' + chatreactAdmin.strings.saving);
            
            // Save each assignment
            var saved = 0;
            var errors = 0;
            
            assignments.forEach(function(assignment) {
                $.ajax({
                    url: chatreactAdmin.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'chatreact_save_assignment',
                        nonce: chatreactAdmin.nonce,
                        post_id: assignment.post_id,
                        chatbot_id: assignment.chatbot_id,
                        exclude_urls: assignment.exclude_urls
                    },
                    success: function(response) {
                        if (response.success) {
                            saved++;
                            // Replace row with saved version
                            var newRowHtml = buildAssignmentRow(response.data.assignment);
                            assignment.$row.replaceWith(newRowHtml);
                        } else {
                            errors++;
                        }
                    },
                    error: function() {
                        errors++;
                    },
                    complete: function() {
                        // Check if all done
                        if (saved + errors === assignments.length) {
                            $saveBtn.prop('disabled', false).html(originalHtml);
                            
                            if (errors === 0) {
                                showNotice('success', chatreactAdmin.strings.saved);
                                $saveBtn.hide();
                                
                                // Update badge
                                var count = $tbody.find('tr').length;
                                $('.chatreact-tab[data-tab="chat"] .chatreact-tab-badge').text(count);
                            } else {
                                showNotice('error', chatreactAdmin.strings.error);
                            }
                        }
                    }
                });
            });
        });
    }

    // Post Types Settings
    function initPostTypesSettings() {
        var $toggleBtn = $('#chatreact-post-types-toggle');
        var $panel = $('#chatreact-post-types-panel');
        var $list = $('#chatreact-post-types-list');
        var $saveBtn = $('#chatreact-save-post-types');
        var $closeBtn = $('#chatreact-close-post-types');
        var postTypesLoaded = false;
        
        // Toggle panel
        $toggleBtn.on('click', function() {
            if ($panel.is(':visible')) {
                $panel.slideUp(200);
            } else {
                $panel.slideDown(200);
                if (!postTypesLoaded) {
                    loadPostTypes();
                }
            }
        });
        
        // Close button
        $closeBtn.on('click', function() {
            $panel.slideUp(200);
        });
        
        // Load post types
        function loadPostTypes() {
            $list.html('<div class="chatreact-post-types-loading"><span class="chatreact-spinner"></span> Loading...</div>');
            
            $.ajax({
                url: chatreactAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'chatreact_get_post_types',
                    nonce: chatreactAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        postTypesLoaded = true;
                        var html = '';
                        
                        response.data.post_types.forEach(function(pt) {
                            html += '<label class="chatreact-post-type-item">';
                            html += '<input type="checkbox" name="post_types[]" value="' + escapeHtml(pt.name) + '"' + (pt.enabled ? ' checked' : '') + '>';
                            html += '<span class="chatreact-post-type-label">' + escapeHtml(pt.label) + '</span>';
                            html += '<span class="chatreact-post-type-slug">(' + escapeHtml(pt.name) + ')</span>';
                            html += '</label>';
                        });
                        
                        $list.html(html);
                    } else {
                        $list.html('<p class="error">' + (response.data.message || 'Error loading post types') + '</p>');
                    }
                },
                error: function() {
                    $list.html('<p class="error">Error loading post types</p>');
                }
            });
        }
        
        // Save post types
        $saveBtn.on('click', function() {
            var $btn = $(this);
            var originalHtml = $btn.html();
            var selectedTypes = [];
            
            $list.find('input[name="post_types[]"]:checked').each(function() {
                selectedTypes.push($(this).val());
            });
            
            $btn.prop('disabled', true).html('<span class="chatreact-spinner"></span>');
            
            $.ajax({
                url: chatreactAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'chatreact_save_post_types',
                    nonce: chatreactAdmin.nonce,
                    post_types: JSON.stringify(selectedTypes)
                },
                success: function(response) {
                    if (response.success) {
                        showNotice('success', response.data.message);
                        $panel.slideUp(200);
                    } else {
                        showNotice('error', response.data.message || 'Error saving');
                    }
                },
                error: function() {
                    showNotice('error', 'Error saving post types');
                },
                complete: function() {
                    $btn.prop('disabled', false).html(originalHtml);
                }
            });
        });
    }

    // Initialize
    $(document).ready(function() {
        initTabs();
        initSearchableSelect();
        initSaveAssignment();
        initEditAssignment();
        initDeleteAssignment();
        initAddRow();
        initSaveAll();
        initPostTypesSettings();
    });

})(jQuery);
