<?php
/**
 * ChatReact Gutenberg Blocks
 *
 * @package ChatReact
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * ChatReact Blocks class
 */
class ChatReact_Blocks {

    /**
     * Single instance of the class
     *
     * @var ChatReact_Blocks|null
     */
    private static $instance = null;

    /**
     * Get single instance of the class
     *
     * @return ChatReact_Blocks
     */
    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action( 'init', array( $this, 'register_blocks' ) );
        add_action( 'init', array( $this, 'set_script_translations' ), 20 );
    }

    /**
     * Set script translations for Gutenberg blocks
     */
    public function set_script_translations() {
        $build_dir = CHATREACT_PLUGIN_DIR . 'build/';
        
        if ( ! file_exists( $build_dir ) ) {
            return;
        }

        // Set translations for each block script
        $blocks = array( 'chat-widget', 'form-widget', 'faq-widget' );
        
        foreach ( $blocks as $block ) {
            $script_handle = 'chatreact-' . $block . '-editor-script';
            wp_set_script_translations(
                $script_handle,
                'chatreact',
                CHATREACT_PLUGIN_DIR . 'languages'
            );
        }
    }

    /**
     * Register Gutenberg blocks
     */
    public function register_blocks() {
        // Check if build directory exists
        $build_dir = CHATREACT_PLUGIN_DIR . 'build/';
        
        if ( ! file_exists( $build_dir ) ) {
            // Development mode - blocks not built yet
            return;
        }

        // Register Chat Widget block
        if ( file_exists( $build_dir . 'chat-widget' ) ) {
            register_block_type(
                $build_dir . 'chat-widget',
                array(
                    'render_callback' => array( $this, 'render_chat_block' ),
                )
            );
        }

        // Register Form Widget block
        if ( file_exists( $build_dir . 'form-widget' ) ) {
            register_block_type(
                $build_dir . 'form-widget',
                array(
                    'render_callback' => array( $this, 'render_form_block' ),
                )
            );
        }

        // Register FAQ Widget block
        if ( file_exists( $build_dir . 'faq-widget' ) ) {
            register_block_type(
                $build_dir . 'faq-widget',
                array(
                    'render_callback' => array( $this, 'render_faq_block' ),
                )
            );
        }
    }

    /**
     * Render Chat Widget block
     *
     * @param array $attributes Block attributes
     * @return string HTML output
     */
    public function render_chat_block( $attributes ) {
        $chatbot_id = isset( $attributes['chatbotId'] ) ? $attributes['chatbotId'] : '';

        if ( empty( $chatbot_id ) ) {
            if ( current_user_can( 'manage_options' ) ) {
                return '<p style="color: #dc2626; background: #fef2f2; padding: 10px; border-radius: 4px;">' 
                    . esc_html__( 'ChatReact: Please configure a Chatbot ID in the block settings.', 'chatreact' ) 
                    . '</p>';
            }
            return '';
        }

        return ChatReact::render_chat_widget( $chatbot_id );
    }

    /**
     * Render Form Widget block
     *
     * @param array $attributes Block attributes
     * @return string HTML output
     */
    public function render_form_block( $attributes ) {
        $form_id = isset( $attributes['formId'] ) ? $attributes['formId'] : '';

        if ( empty( $form_id ) ) {
            if ( current_user_can( 'manage_options' ) ) {
                return '<p style="color: #dc2626; background: #fef2f2; padding: 10px; border-radius: 4px;">' 
                    . esc_html__( 'ChatReact: Please configure a Form ID in the block settings.', 'chatreact' ) 
                    . '</p>';
            }
            return '';
        }

        return ChatReact::render_form_widget(
            $form_id,
            array(
                'language' => isset( $attributes['language'] ) ? $attributes['language'] : '',
            )
        );
    }

    /**
     * Render FAQ Widget block
     *
     * @param array $attributes Block attributes
     * @return string HTML output
     */
    public function render_faq_block( $attributes ) {
        $chatbot_id = isset( $attributes['chatbotId'] ) ? $attributes['chatbotId'] : '';

        if ( empty( $chatbot_id ) ) {
            if ( current_user_can( 'manage_options' ) ) {
                return '<p style="color: #dc2626; background: #fef2f2; padding: 10px; border-radius: 4px;">' 
                    . esc_html__( 'ChatReact: Please configure a Chatbot ID in the block settings.', 'chatreact' ) 
                    . '</p>';
            }
            return '';
        }

        return ChatReact::render_faq_widget(
            $chatbot_id,
            array(
                'categories' => isset( $attributes['categories'] ) ? $attributes['categories'] : '',
                'container'  => isset( $attributes['containerId'] ) ? $attributes['containerId'] : '',
                'language'   => isset( $attributes['language'] ) ? $attributes['language'] : '',
            )
        );
    }
}
