<?php
/**
 * Main ChatReact Plugin Class
 *
 * @package ChatReact
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * ChatReact main class
 */
class ChatReact {

    /**
     * Single instance of the class
     *
     * @var ChatReact|null
     */
    private static $instance = null;

    /**
     * Get single instance of the class
     *
     * @return ChatReact
     */
    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        // Constructor is private to enforce singleton pattern
    }

    /**
     * Get the API URL
     *
     * @return string
     */
    public static function get_api_url() {
        return CHATREACT_API_URL;
    }

    /**
     * Render a chat widget
     *
     * All widget settings (position, colors, etc.) are configured in the ChatReact dashboard.
     *
     * @param string $chatbot_id The chatbot ID
     * @return string HTML output
     */
    public static function render_chat_widget( $chatbot_id ) {
        if ( empty( $chatbot_id ) ) {
            return '';
        }

        // Add version parameter to prevent caching issues
        return sprintf(
            '<script src="%s/embed/widget.js?v=%s" data-chatbot-id="%s" async></script>',
            esc_url( self::get_api_url() ),
            CHATREACT_VERSION,
            esc_attr( $chatbot_id )
        );
    }

    /**
     * Render a form widget
     *
     * @param string $form_id The form ID
     * @param array  $options Optional widget options
     * @return string HTML output
     */
    public static function render_form_widget( $form_id, $options = array() ) {
        if ( empty( $form_id ) ) {
            return '';
        }

        static $instance = 0;
        $instance++;

        $container_id = 'chatreact-form-' . $instance;

        $defaults = array(
            'language' => '',
        );

        $options = wp_parse_args( $options, $defaults );

        // Build data attributes
        $data_attrs = sprintf( 'data-form-id="%s"', esc_attr( $form_id ) );

        if ( ! empty( $options['language'] ) ) {
            $data_attrs .= sprintf( ' data-language="%s"', esc_attr( $options['language'] ) );
        }

        // Add version parameter to prevent caching issues
        return sprintf(
            '<div id="%s"></div><script src="%s/embed/form.js?v=%s" %s data-container="%s" async></script>',
            esc_attr( $container_id ),
            esc_url( self::get_api_url() ),
            CHATREACT_VERSION,
            $data_attrs,
            esc_attr( $container_id )
        );
    }

    /**
     * Render an FAQ widget
     *
     * @param string $chatbot_id The chatbot ID
     * @param array  $options    Optional widget options
     * @return string HTML output
     */
    public static function render_faq_widget( $chatbot_id, $options = array() ) {
        if ( empty( $chatbot_id ) ) {
            return '';
        }

        static $instance = 0;
        $instance++;

        $defaults = array(
            'container'  => '',
            'categories' => '',
            'language'   => '',
        );

        $options = wp_parse_args( $options, $defaults );

        // Use custom container or generate one
        $container_id = ! empty( $options['container'] ) 
            ? $options['container'] 
            : 'chatreact-faq-' . $instance;

        // Build data attributes
        $data_attrs = sprintf( 'data-chatbot-id="%s"', esc_attr( $chatbot_id ) );
        $data_attrs .= sprintf( ' data-container="%s"', esc_attr( $container_id ) );

        if ( ! empty( $options['categories'] ) ) {
            $data_attrs .= sprintf( ' data-categories="%s"', esc_attr( $options['categories'] ) );
        }

        if ( ! empty( $options['language'] ) ) {
            $data_attrs .= sprintf( ' data-lang="%s"', esc_attr( $options['language'] ) );
        }

        // Add version parameter to prevent caching issues
        return sprintf(
            '<div id="%s"></div><script src="%s/embed/faq-widget.js?v=%s" %s async></script>',
            esc_attr( $container_id ),
            esc_url( self::get_api_url() ),
            CHATREACT_VERSION,
            $data_attrs
        );
    }
}
