<?php
/**
 * ChatReact Meta Boxes
 *
 * @package ChatReact
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * ChatReact Meta Boxes class
 */
class ChatReact_Meta_Boxes {

    /**
     * Single instance of the class
     *
     * @var ChatReact_Meta_Boxes|null
     */
    private static $instance = null;

    /**
     * Meta key for storing widget settings
     *
     * @var string
     */
    const META_KEY = '_chatreact_widget';

    /**
     * Get single instance of the class
     *
     * @return ChatReact_Meta_Boxes
     */
    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action( 'add_meta_boxes', array( $this, 'add_meta_boxes' ) );
        add_action( 'save_post', array( $this, 'save_meta_box' ), 10, 2 );
        add_action( 'wp_footer', array( $this, 'render_page_widget' ) );
    }

    /**
     * Add meta boxes
     */
    public function add_meta_boxes() {
        $post_types = $this->get_supported_post_types();

        foreach ( $post_types as $post_type ) {
            add_meta_box(
                'chatreact_widget',
                __( 'ChatReact Widget', 'chatreact' ),
                array( $this, 'render_meta_box' ),
                $post_type,
                'side',
                'default'
            );
        }
    }

    /**
     * Get supported post types
     *
     * @return array
     */
    private function get_supported_post_types() {
        $post_types = get_post_types( array( 'public' => true ), 'names' );
        
        // Remove attachment
        unset( $post_types['attachment'] );

        /**
         * Filter supported post types for ChatReact meta boxes
         *
         * @param array $post_types Array of post type names
         */
        return apply_filters( 'chatreact_meta_box_post_types', $post_types );
    }

    /**
     * Render meta box content
     *
     * @param WP_Post $post The post object
     */
    public function render_meta_box( $post ) {
        // Get saved values
        $settings = get_post_meta( $post->ID, self::META_KEY, true );
        $settings = wp_parse_args(
            is_array( $settings ) ? $settings : array(),
            array(
                'enabled'    => '',
                'chatbot_id' => '',
                'position'   => '',
                'disabled'   => '',
            )
        );

        // Nonce field
        wp_nonce_field( 'chatreact_meta_box', 'chatreact_meta_box_nonce' );
        ?>
        <div class="chatreact-meta-box">
            <p>
                <label>
                    <input 
                        type="checkbox" 
                        name="chatreact_widget[enabled]" 
                        value="1" 
                        <?php checked( $settings['enabled'], '1' ); ?>
                    >
                    <?php esc_html_e( 'Enable custom widget for this page', 'chatreact' ); ?>
                </label>
            </p>

            <div class="chatreact-meta-box-fields" style="<?php echo empty( $settings['enabled'] ) ? 'display:none;' : ''; ?>">
                <p>
                    <label for="chatreact_chatbot_id">
                        <?php esc_html_e( 'Chatbot ID:', 'chatreact' ); ?>
                    </label>
                    <input 
                        type="text" 
                        id="chatreact_chatbot_id" 
                        name="chatreact_widget[chatbot_id]" 
                        value="<?php echo esc_attr( $settings['chatbot_id'] ); ?>" 
                        class="widefat"
                    >
                </p>

                <p>
                    <label for="chatreact_position">
                        <?php esc_html_e( 'Position:', 'chatreact' ); ?>
                    </label>
                    <select id="chatreact_position" name="chatreact_widget[position]" class="widefat">
                        <option value="" <?php selected( $settings['position'], '' ); ?>>
                            <?php esc_html_e( 'Default', 'chatreact' ); ?>
                        </option>
                        <option value="bottom-right" <?php selected( $settings['position'], 'bottom-right' ); ?>>
                            <?php esc_html_e( 'Bottom Right', 'chatreact' ); ?>
                        </option>
                        <option value="bottom-left" <?php selected( $settings['position'], 'bottom-left' ); ?>>
                            <?php esc_html_e( 'Bottom Left', 'chatreact' ); ?>
                        </option>
                    </select>
                </p>
            </div>

            <hr>

            <p>
                <label>
                    <input 
                        type="checkbox" 
                        name="chatreact_widget[disabled]" 
                        value="1" 
                        <?php checked( $settings['disabled'], '1' ); ?>
                    >
                    <?php esc_html_e( 'Disable all widgets on this page', 'chatreact' ); ?>
                </label>
            </p>
        </div>

        <script>
        jQuery(document).ready(function($) {
            $('input[name="chatreact_widget[enabled]"]').on('change', function() {
                if ($(this).is(':checked')) {
                    $('.chatreact-meta-box-fields').slideDown();
                } else {
                    $('.chatreact-meta-box-fields').slideUp();
                }
            });
        });
        </script>
        <?php
    }

    /**
     * Save meta box data
     *
     * @param int     $post_id The post ID
     * @param WP_Post $post    The post object
     */
    public function save_meta_box( $post_id, $post ) {
        // Check nonce
        if ( ! isset( $_POST['chatreact_meta_box_nonce'] ) || 
             ! wp_verify_nonce( $_POST['chatreact_meta_box_nonce'], 'chatreact_meta_box' ) ) {
            return;
        }

        // Check autosave
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        // Check permissions
        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        // Check if this is a supported post type
        if ( ! in_array( $post->post_type, $this->get_supported_post_types(), true ) ) {
            return;
        }

        // Get and sanitize data
        $widget_data = isset( $_POST['chatreact_widget'] ) ? $_POST['chatreact_widget'] : array();

        $settings = array(
            'enabled'    => ! empty( $widget_data['enabled'] ) ? '1' : '',
            'chatbot_id' => sanitize_text_field( $widget_data['chatbot_id'] ?? '' ),
            'position'   => sanitize_text_field( $widget_data['position'] ?? '' ),
            'disabled'   => ! empty( $widget_data['disabled'] ) ? '1' : '',
        );

        // Save or delete meta
        if ( empty( $settings['enabled'] ) && empty( $settings['disabled'] ) ) {
            delete_post_meta( $post_id, self::META_KEY );
        } else {
            update_post_meta( $post_id, self::META_KEY, $settings );
        }
    }

    /**
     * Render page-specific widget in footer
     */
    public function render_page_widget() {
        if ( ! is_singular() ) {
            return;
        }

        $post_id  = get_the_ID();
        $settings = get_post_meta( $post_id, self::META_KEY, true );

        if ( ! is_array( $settings ) ) {
            return;
        }

        // If widgets are disabled for this page, add a flag
        if ( ! empty( $settings['disabled'] ) ) {
            // We could add a JavaScript flag to prevent any widgets from loading
            // For now, we just don't render anything
            return;
        }

        // If custom widget is enabled, render it
        if ( ! empty( $settings['enabled'] ) && ! empty( $settings['chatbot_id'] ) ) {
            echo ChatReact::render_chat_widget( // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
                $settings['chatbot_id'],
                array(
                    'position' => $settings['position'] ?? '',
                )
            );
        }
    }

    /**
     * Check if widgets are disabled for a specific post
     *
     * @param int $post_id The post ID
     * @return bool
     */
    public static function is_disabled( $post_id ) {
        $settings = get_post_meta( $post_id, self::META_KEY, true );
        return is_array( $settings ) && ! empty( $settings['disabled'] );
    }
}
