<?php
/**
 * ChatReact Shortcodes
 *
 * @package ChatReact
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * ChatReact Shortcodes class
 */
class ChatReact_Shortcodes {

    /**
     * Single instance of the class
     *
     * @var ChatReact_Shortcodes|null
     */
    private static $instance = null;

    /**
     * Get single instance of the class
     *
     * @return ChatReact_Shortcodes
     */
    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->register_shortcodes();
    }

    /**
     * Register all shortcodes
     */
    private function register_shortcodes() {
        add_shortcode( 'chatreact', array( $this, 'render_chat_shortcode' ) );
        add_shortcode( 'chatreact_form', array( $this, 'render_form_shortcode' ) );
        add_shortcode( 'chatreact_faq', array( $this, 'render_faq_shortcode' ) );
    }

    /**
     * Render chat widget shortcode
     *
     * Usage: [chatreact id="CHATBOT_ID"]
     * All widget settings (position, colors, etc.) are configured in your ChatReact dashboard.
     *
     * @param array $atts Shortcode attributes
     * @return string HTML output
     */
    public function render_chat_shortcode( $atts ) {
        $atts = shortcode_atts(
            array(
                'id' => '',
            ),
            $atts,
            'chatreact'
        );

        if ( empty( $atts['id'] ) ) {
            if ( current_user_can( 'manage_options' ) ) {
                return '<p style="color: #dc2626; background: #fef2f2; padding: 10px; border-radius: 4px;">' 
                    . esc_html__( 'ChatReact: Please provide a chatbot ID. Example: [chatreact id="your-chatbot-id"]', 'chatreact' ) 
                    . '</p>';
            }
            return '';
        }

        return ChatReact::render_chat_widget( $atts['id'] );
    }

    /**
     * Render form widget shortcode
     *
     * Usage: [chatreact_form id="FORM_ID" language="de"]
     *
     * @param array $atts Shortcode attributes
     * @return string HTML output
     */
    public function render_form_shortcode( $atts ) {
        $atts = shortcode_atts(
            array(
                'id'       => '',
                'language' => '',
            ),
            $atts,
            'chatreact_form'
        );

        if ( empty( $atts['id'] ) ) {
            if ( current_user_can( 'manage_options' ) ) {
                return '<p style="color: #dc2626; background: #fef2f2; padding: 10px; border-radius: 4px;">' 
                    . esc_html__( 'ChatReact: Please provide a form ID. Example: [chatreact_form id="your-form-id"]', 'chatreact' ) 
                    . '</p>';
            }
            return '';
        }

        return ChatReact::render_form_widget(
            $atts['id'],
            array(
                'language' => $atts['language'],
            )
        );
    }

    /**
     * Render FAQ widget shortcode
     *
     * Usage: [chatreact_faq id="CHATBOT_ID" categories="general,pricing" container="my-faq" language="de"]
     *
     * @param array $atts Shortcode attributes
     * @return string HTML output
     */
    public function render_faq_shortcode( $atts ) {
        $atts = shortcode_atts(
            array(
                'id'         => '',
                'categories' => '',
                'container'  => '',
                'language'   => '',
            ),
            $atts,
            'chatreact_faq'
        );

        if ( empty( $atts['id'] ) ) {
            if ( current_user_can( 'manage_options' ) ) {
                return '<p style="color: #dc2626; background: #fef2f2; padding: 10px; border-radius: 4px;">' 
                    . esc_html__( 'ChatReact: Please provide a chatbot ID. Example: [chatreact_faq id="your-chatbot-id"]', 'chatreact' ) 
                    . '</p>';
            }
            return '';
        }

        return ChatReact::render_faq_widget(
            $atts['id'],
            array(
                'categories' => $atts['categories'],
                'container'  => $atts['container'],
                'language'   => $atts['language'],
            )
        );
    }
}
